<?php

namespace App\Filament\Resources;

use App\Filament\Resources\VideosResource\Pages;
use App\Filament\Resources\VideosResource\RelationManagers;
use App\Models\Videos;
use Filament\Forms;
use Filament\Forms\Components\Actions\Action;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;

class VideosResource extends Resource
{
    protected static ?string $model = Videos::class;
    protected static ?string $navigationGroup = 'Conteúdo';
    protected static ?string $navigationIcon = 'heroicon-c-play-circle';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([

                Forms\Components\FileUpload::make('capa')
                    ->label('Capa do video')
                    ->disk('public')
                    ->image()
                    ->directory('videos/capas')
                    ->downloadable(),

                Forms\Components\FileUpload::make('link')
                    ->label('Upload do video')
                    ->disk('public')
                    ->directory('videos/videos')
                    ->downloadable(),
                Forms\Components\TextInput::make('titulo')
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true) // só dispara quando sai do campo
                    ->afterStateUpdated(function (string $state, Set $set, Get $get) {
                        // Preenche o slug apenas se ainda estiver vazio
                        if (blank($get('slug')) && filled($state)) {
                            $set('slug', Str::slug($state));
                        }
                    }),
                Forms\Components\TextInput::make('slug')
                    ->required()
                    ->maxLength(255),
                Forms\Components\Select::make('categorias')
                    ->label('Categorias')
                    ->relationship('categorias', 'nome') // M:N
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->native(false)

                    // CRIAR em MODAL
                    ->createOptionForm([
                        Forms\Components\TextInput::make('nome')
                            ->label('Nome da categoria')
                            ->maxLength(150)
                            ->required(),
                    ])
                    ->createOptionAction(function (Action $action) {
                        return $action
                            ->modalHeading('Nova categoria')
                            ->modalSubmitActionLabel('Salvar')
                            ->modalWidth('md')
                            ->slideOver()
                            ->successNotificationTitle('Categoria criada com sucesso');
                    })

                    // EDITAR em MODAL (opcional)
                    ->editOptionForm([
                        Forms\Components\TextInput::make('nome')
                            ->label('Nome da categoria')
                            ->maxLength(150)
                            ->required(),
                    ])
                    ->editOptionAction(function (Action $action) {
                        return $action
                            ->modalHeading('Editar categoria')
                            ->modalSubmitActionLabel('Atualizar')
                            ->modalWidth('md')
                            ->slideOver()
                            ->successNotificationTitle('Categoria atualizada');
                    }),
                Forms\Components\RichEditor::make('texto')
                    ->label('Texto')
                    ->columnSpanFull(),
            ])->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('titulo')
                    ->searchable(),
                Tables\Columns\TextColumn::make('descricao')
                    ->searchable(),
                Tables\Columns\TextColumn::make('link')
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->searchable(),
                Tables\Columns\TextColumn::make('capa')
                    ->searchable(),
                Tables\Columns\TextColumn::make('visualizacoes')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListVideos::route('/'),
            'create' => Pages\CreateVideos::route('/create'),
            'edit' => Pages\EditVideos::route('/{record}/edit'),
        ];
    }
}
