<?php

namespace App\Filament\Resources;

use App\Filament\Resources\NoticiasResource\Pages;

use App\Filament\Resources\NoticiasResource\RelationManagers\ImagensRelationManager;
use App\Models\Noticia;
use Filament\Forms\Components\Actions\Action;
use Illuminate\Support\Str;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Get;
use Filament\Forms\Set;


class NoticiasResource extends Resource
{


    protected static ?string $model = Noticia::class;

    protected static ?string $navigationIcon = 'heroicon-o-newspaper';
    protected static ?string $navigationGroup = 'Conteúdo';
    protected static ?string $modelLabel = 'Notícia';
    protected static ?string $pluralModelLabel = 'Notícias';
    protected static ?string $navigationLabel = 'Notícias';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make('Dados da notícia')
                ->schema([
                    Forms\Components\Grid::make(12)->schema([
                        // Coluna ESQUERDA (8/12)
                        Forms\Components\Group::make([
                            Forms\Components\TextInput::make('titulo')
                                ->label('Título')
                                ->maxLength(255)
                                ->required()
                                ->live(onBlur: true) // só dispara quando sai do campo
                                ->afterStateUpdated(function (string $state, Set $set, Get $get) {
                                    // Preenche o slug apenas se ainda estiver vazio
                                    if (blank($get('slug')) && filled($state)) {
                                        $set('slug', Str::slug($state));
                                    }
                                }),

                            Forms\Components\TextInput::make('subtitulo')
                                ->label('Subtítulo')
                                ->maxLength(255),

                            Forms\Components\TextInput::make('slug')
                                ->label('Slug')
                                ->disabled()     // usuário não edita
                                ->dehydrated()   // mas o valor é salvo
                                ->required()

                        ])->columnSpan(8),

                        // Coluna DIREITA (4/12) -> Upload da capa
                        Forms\Components\Group::make([
                            Forms\Components\FileUpload::make('capa')
                                ->label('Capa')
                                ->image()
                                ->disk('public')
                                ->directory('noticias/capas')
                                ->imageEditor()
                                ->downloadable(),
                        ])->columnSpan(4),

                        // Abaixo das duas colunas, ocupando toda a largura
                        Forms\Components\RichEditor::make('texto')
                            ->label('Texto')
                            ->columnSpan(12),
                    ]),
                ]),

            // Mantém as vinculações (pode ficar abaixo)
            Forms\Components\Section::make('Vinculações')
                ->schema([
                    Forms\Components\Select::make('secretarias_id')
                        ->label('Secretaria')
                        ->relationship('secretaria', 'nome')
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->native(false),

                    Forms\Components\Select::make('categorias')
                        ->label('Categorias')
                        ->relationship('categorias', 'nome') // M:N
                        ->multiple()
                        ->preload()
                        ->searchable()
                        ->native(false)

                        // CRIAR em MODAL
                        ->createOptionForm([
                            Forms\Components\TextInput::make('nome')
                                ->label('Nome da categoria')
                                ->maxLength(150)
                                ->required(),
                        ])
                        ->createOptionAction(function (Action $action) {
                            return $action
                                ->modalHeading('Nova categoria')
                                ->modalSubmitActionLabel('Salvar')
                                ->modalWidth('md')
                                ->slideOver()
                                ->successNotificationTitle('Categoria criada com sucesso');
                        })

                        // EDITAR em MODAL (opcional)
                        ->editOptionForm([
                            Forms\Components\TextInput::make('nome')
                                ->label('Nome da categoria')
                                ->maxLength(150)
                                ->required(),
                        ])
                        ->editOptionAction(function (Action $action) {
                            return $action
                                ->modalHeading('Editar categoria')
                                ->modalSubmitActionLabel('Atualizar')
                                ->modalWidth('md')
                                ->slideOver()
                                ->successNotificationTitle('Categoria atualizada');
                        }),
                ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('titulo')
                    ->label('Título')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('categorias.nome')
                    ->label('Categorias')
                    ->badge()
                    ->separator(', ')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('secretaria.nome')
                    ->label('Secretaria')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Criado em')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('categorias')
                    ->label('Categoria')
                    ->relationship('categorias', 'nome'),

                Tables\Filters\SelectFilter::make('secretaria')
                    ->label('Secretaria')
                    ->relationship('secretaria', 'nome'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListNoticias::route('/'),
            'create' => Pages\CreateNoticias::route('/create'),
            'edit'   => Pages\EditNoticias::route('/{record}/edit'),
        ];
    }

    public static function getRelations(): array
    {
        return [ImagensRelationManager::class];
    }


}
